import puppeteer from 'puppeteer';
import user_agent from 'user-agents';
import { HtmlOutput } from 'referrers/html-output';
import { IBrowserError } from '../interfaces/browser/error';
import { SearchLookup } from 'referrers/search-input-lookup';
import _ from 'lodash';
import * as Consts from '../shared/constants';
import * as utils from '../shared/utils';
import { ClickSearchElement, IClickSearchElementError } from 'referrers/click-search-element';
import * as puppeteer_utils from '../shared/browser-utils'
import { resolve } from 'path';
import * as BrowserUtils from '../shared/browser-utils';

export const getHTMLContent = async(browser: puppeteer.Browser, url: string, timeout: number) : Promise<HtmlOutput> =>  {

	return new Promise(async (resolve, reject) => {

		const pie_response = {					
			url: '',
			statusCode: -1,
			responseHeaders: undefined,				
			requestsChain: {},                        		
			html: undefined,
			requestInterceptions: [],
			responseInterceptions: [],
			hyperlinks: [],					
			status: undefined,
			exceptions: []    
		} as HtmlOutput
	
		const executionContexts = ['execution_onload'];
				
		try 
		{		       
			pie_response['url'] = url;
			const rootDomainRegex = utils.generateRootDomainRegex(url);		
			const page = await browser.newPage();     

			// handling page errors.
			page.on('error', e => {
				throw e;
			});
	
			if(timeout) {		
				page.setDefaultNavigationTimeout(timeout);
			}
			
			const userAgent = new user_agent({ deviceCategory: 'desktop' });					
			await page.setUserAgent(userAgent.toString());	
			await page.setViewport({width: userAgent.data.screenWidth, height: userAgent.data.screenHeight});
			
			// note: add trailing slash since chrome adds it
			if (!url.endsWith('/'))
				url = url + '/';
	
			// urls hold redirect chain
			let requestsChain: {[key: string]: any} = {};
			const requestInterceptions: Array<{[key: string]: any}> = [];
			const responseInterceptions: Array<{[key: string]: any}> = [];
	
			const client = await page.target().createCDPSession();

			await client.send('Network.enable');
			
			client.on('Network.requestWillBeSent', (e) => {
				
				// if (e.type !== 'Document') {
				// 	return;
				// }
				
				const requestId = e.requestId;

				if(!requestsChain[requestId])
					requestsChain[requestId] = {};

				// check if url redirected
				if (typeof e.redirectResponse != 'undefined') {
					
					if(!requestsChain[requestId]['redirects'])
						requestsChain[requestId]['redirects'] = [];

						requestsChain[requestId]['redirects'].push({
						redirect: true,
						//requestId: e.requestId,
						status: e.redirectResponse.status,
						url: e.request.url,
						loader: e.loaderId,
						documentURL: e.documentURL,
						//initiator: e.initiator,
						type: e.type,
						frameId: e.frameId
						//timestamp: e.timestamp,
						//wallTime: e.wallTime,
						//hasUserGesture: e.hasUserGesture
					});

				} else {
				
					requestsChain[requestId] = {
							redirect: false,
							//requestId: e.requestId,							
							url: e.request.url,
							//loader: e.loaderId,
							documentURL: e.documentURL,
							//initiator: e.initiator,
							type: e.type,
							frameId: e.frameId
							//timestamp: e.timestamp,
							//wallTime: e.wallTime,	
							//hasUserGesture: e.hasUserGesture							
					};
				}
			});
			
			await client.send('Debugger.enable');												
			await page.setRequestInterception(true);
					
			page.on('request', request => {
	
				try 
				{    
					const url = request.url();        
					const resourceType = request.resourceType();  
					if(!url.match(rootDomainRegex)) {   		        								
					requestInterceptions.push({
							'context': _.last(executionContexts),
							'url': url,
							'headers': request.headers(),
							'method': request.method(),
							'resourceType': resourceType,							
							'payload': request.postData()
							
						});
					}             
									
				request.continue();
	
				} catch (e) {					
				}
	
			});
				
			page.on('response', async resp => {

				try 
				{    
					const url = resp.url();	
					
					if(!url.match(rootDomainRegex)) {    
						
						let resourceType = resp.request().resourceType();
						let method = resp.request().method();				
						let headers = undefined;	
						let payload = undefined;	
						
						try {

							headers = await resp.headers();	
							if(!Consts.REFERRERS_CRAWLER.IGNORED_PAYLOADS_BY_RESOURCE_TYPES.includes(resourceType))
							{
								payload = await resp.text();														
							}
							else
							{
								payload = null;
							}
								
						} catch {}	

						responseInterceptions.push({
							'context': _.last(executionContexts),
							'url': url,						
							'headers': headers,
							'method': method,
							'resourceType': resourceType, 														
							'payload': payload										
						});
					}					
				}
				catch(e){						
				}
			});
			
			let response = undefined;

			try {	

				try {
					response = await page.goto(url, {waitUntil: 'networkidle2'}); 									
				} catch(e) {				
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE_NETWORK_IDLE, message: e.message, stack: e.stack } as IBrowserError);					
					requestsChain = {};
					requestInterceptions.splice(0, requestInterceptions.length);
					responseInterceptions.splice(0, responseInterceptions.length);					
				}
				finally {					
					if(!response)
						response = await page.goto(url);
				}
				
				pie_response['html'] = await page.content();	

				if(response)
				{						
					pie_response['statusCode'] = response.status();
					pie_response['responseHeaders'] = response.headers();	
				}
				else
					throw(new Error(`HTTP Response is undefined.`))
				
				pie_response['requestsChain'] = requestsChain;									
				pie_response['requestInterceptions'] = requestInterceptions;
				pie_response['responseInterceptions'] = responseInterceptions;	
										
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE, message: e.message, stack: e.stack } as IBrowserError);															
				throw(e);
			}

			try {
					
				executionContexts.push('execution_collectingHyperlinks');
				const hyperlinks = await page.$$eval('a', e => e.map((a: HTMLAnchorElement)=>{
					return {
					 id: a.getAttribute('id'),
					 name: a.getAttribute('name'), 					  
					 className: a.getAttribute('class'),
					 href: a.getAttribute('href'),
					 innerText: a.innerText.replace(/^\s+|\s+$/gm,''),
					 rel: a.getAttribute('rel'),
					 referrerpolicy: a.getAttribute('referrerpolicy')
				   };
			   }));

			   if(hyperlinks && hyperlinks.length > 0)
			   		pie_response['hyperlinks'] = hyperlinks;
			  
			} catch (e) {				
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACTING_LINKS, message: e.message, stack: e.stack } as IBrowserError);			
			}
											
			try {						

				executionContexts.push('execution_closingPage');	
				await page.close();
			
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLOSING_PAGE, message: e.message, stack: e.stack } as IBrowserError);
			}		
			         					
			pie_response.status = { error: 0 };		
						
		} catch (e) { 	
			pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.HTML_STEP_EXCEPTION, message: e.message, stack: e.stack } as IBrowserError);    
			pie_response.status = { error: 1, message: e.message, stack: e.stack, context: _.last(executionContexts) };		
		}
		finally {
			return resolve(pie_response);  
		}
	
	});
}

export const searchElementLookup = async(browser: puppeteer.Browser, url: string, timeout: number, languageCodes: Array<string>) : Promise<SearchLookup> => {

	return new Promise(async (resolve, reject) => {

		const pie_response = {
			url,
			searchInputs: [],
			alternatives: [],			
			status: undefined,
			exceptions: []			  
		} as SearchLookup

		const executionContexts = ['execution_onload'];
		
		try 
		{			
			pie_response['url'] = url;

			const page = await browser.newPage(); 
			
			// handling page errors.
			page.on('error', e => {
				throw e;
			});
			
			if(timeout) {
				page.setDefaultNavigationTimeout(timeout);
			}
			
			const userAgent = new user_agent({ deviceCategory: 'desktop' });			
			await page.setUserAgent(userAgent.toString());
			await page.setViewport({width: userAgent.data.screenWidth, height: userAgent.data.screenHeight});
		
			let response = undefined;
			
			try {	

				try {
					response = await page.goto(url, {waitUntil: 'networkidle2'}); 									
				} catch(e) {				
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE_NETWORK_IDLE, message: e.message, stack: e.stack } as IBrowserError);							
				}
				finally {					
					if(!response)
						response = await page.goto(url);
				}
															
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE, message: e.message, stack: e.stack } as IBrowserError);		
				throw(e);
			}
							 
			executionContexts.push('execution_settingSearchInputs');
			const searchRelatedElements: Array<string> = [];

			try {				
				languageCodes.forEach(code => {
					const search_term: string = Consts.REFERRERS_CRAWLER.LANGUAGE_MAPPINGS.search_terms[code];
					if(search_term && search_term.length > 0)
						searchRelatedElements.push.apply(searchRelatedElements, Consts.REFERRERS_CRAWLER.SEARCH_QUERY_SELECTORS.map((d: string) => d.replace(/search/g, search_term)));
				});

			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SETUP_LANGUAGE_FOR_SEARCH_LOOKUPS, message: e.message, stack: e.stack } as IBrowserError);		
				throw(e);				
			}

			executionContexts.push('execution_searchingInputs');

			for(let i = 0; i < searchRelatedElements.length ; i++ ) {    

				try 
				{               					                             
					const searchRelatedElementSelector = searchRelatedElements[i];					                             
							
					let inputs: any = [];
					
						if(searchRelatedElementSelector.startsWith('input'))
						{      
							try {
								
								inputs = await page.$$eval(searchRelatedElementSelector, e=>e.map((el: HTMLElement)=>{
									return {
										id: el.getAttribute('id'),     
										type: el.getAttribute('type') || (el as any).type, 
										name: el.getAttribute('name'), 
										placeholder: el.getAttribute('placeholder'), 
										className: el.getAttribute('class'),
										tagName: el.tagName.toLowerCase()
								   };
							   }));
							  
							} catch (e) {
								pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack} as IBrowserError);		
							}         														
						}
						else
						{
							const searchRelatedElement = await page.$(searchRelatedElementSelector); 
								      
							if(searchRelatedElement && searchRelatedElement.$$eval)
							{      								     
								try {
									inputs = await searchRelatedElement.$$eval('input', e=>e.map((el: HTMLInputElement)=>{
										return {
										id: el.getAttribute('id'),     
										type: el.getAttribute('type') || (el as any).type, 
										name: el.getAttribute('name'), 
										placeholder: el.getAttribute('placeholder'), 
										className: el.getAttribute('class'),
										tagName: el.tagName.toLowerCase()
										};
									}));
								} catch (e) {
									pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack} as IBrowserError);		
								}    													 																
							}
						}

						try {
									
							const element = await page.$eval(searchRelatedElementSelector, (el: HTMLElement) => {
								return {
									id: el.getAttribute('id'),     
									type: el.getAttribute('type') || (el as any).type, 
									name: el.getAttribute('name'), 
									placeholder: el.getAttribute('placeholder'), 
									className: el.getAttribute('class'),
									tagName: el.tagName.toLowerCase()
								};
							});
							
							if(element && utils.validateSearchElementProps(element as HTMLInputElement))
							{										
								inputs.push(element);
							}
								

							} catch (e) {
								pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack} as IBrowserError);		
							}  
																				 
						if(inputs && inputs.length > 0)
						{                          
							const searchInputItem = {searchRelatedElementSelector, inputs};
							pie_response['searchInputs'].push(searchInputItem);                                                                                                                                
						}
						else
						{       
							try {
									const remoteTryQuerySelector = searchRelatedElementSelector.startsWith('input') ? searchRelatedElementSelector : searchRelatedElementSelector + ' input';
									let remotely_fetched_inputs = await page.evaluate(async (selector_param: string) => {          
																
										let inputs = document.querySelectorAll(selector_param);
										let inputsNotSet: Array<any> = [];                                
										if(inputs && inputs.length > 0){
											[].forEach.call(inputs, function(el: HTMLElement) {                                                                               
												try {
													inputsNotSet.push({
														id: el.getAttribute('id'), 
														type: el.getAttribute('type') || (el as any).type, 
														name: el.getAttribute('name'), 
														placeholder: el.getAttribute('placeholder'), 
														className: el.getAttribute('class'),
														tagName: el.tagName.toLowerCase()												
												});          
												} catch (e) {
												
												}                                                                                
											});                         
										}                                                                
										return inputsNotSet;

								}, remoteTryQuerySelector);     
		
								if(remotely_fetched_inputs && remotely_fetched_inputs.length > 0)
								{                       
									const searchInputItem = {searchRelatedElementSelector, 'inputs': remotely_fetched_inputs};								
									pie_response['searchInputs'].push(searchInputItem);
								}	

							} catch (e) {
								pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack} as IBrowserError);
							}  																		
						}  
																
				} catch(e) {    															
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack}  as IBrowserError);
				}           
			}

			if(pie_response['searchInputs'].length === 0)
			{
				try {
								
					const alternatives = await page.$$eval(`[href*="${utils.extractRootDomain(url)}"]`, e=>e.map((a: HTMLAnchorElement)=>{
						return {
							id: a.getAttribute('id'),
							name: a.getAttribute('name'), 					  
							className: a.getAttribute('class'),
							href: a.getAttribute('href'),
							innerText: a.innerText.replace(/^\s+|\s+$/gm,''),
							rel: a.getAttribute('rel'),
							referrerpolicy: a.getAttribute('referrerpolicy')				  
					   };
				   }));

				   if(alternatives && alternatives.length > 0)
				   pie_response['alternatives'] = alternatives;
				  
				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack} as IBrowserError);
				}         		
			}
							
			try {						

				executionContexts.push('execution_closingPage');	
				await page.close();
			
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLOSING_PAGE, message: e.message, stack: e.stack } as IBrowserError);
			}		

			pie_response.status = { error: 0 };							
			
		} catch (e) { 	
			pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.LOOKUP_STEP_EXCEPTION, message: e.message, stack: e.stack } as IBrowserError);    		 		
			pie_response.status = { error: 1, message: e.message, stack: e.stack, context: _.last(executionContexts) };			
		}
		finally{
			return resolve(pie_response);  
		}
	});
}

export const clickSearchElement = async(browser: puppeteer.Browser, url: string, searchElement: any, timeout: number, searchQuery: string) : Promise<ClickSearchElement> => {

	return new Promise(async (resolve, reject) => {

		const pie_response = {					
			querySelector: undefined,
			url: '',
			statusCode: -1,
			responseHeaders: undefined,				
			requestsChain: {},     
			listeners: [],
			requestInterceptions: [],
			responseInterceptions: [],
			hyperlinks: [],
			searchElement: undefined,         				
			search_url: null,
			status: undefined,
			exceptions: []
		}  as ClickSearchElement

        const executionContexts = ['execution_prepQuerySelector'];
		let page: puppeteer.Page;
		
		const timeoutId = setTimeout(() => {
            return reject();
        }, 300000);
       
		try {
				
			pie_response['url'] = url;
			pie_response['searchElement'] = searchElement;

			const querySelector = pie_response['querySelector'] =  utils.buildQuerySelector(searchElement,'input');																														

			page = await browser.newPage();
			
			// handling page errors.
			page.on('error', e => {
				throw e;
			});
					
			if(timeout) {					
				page.setDefaultNavigationTimeout(timeout);
			}
			
			const userAgent = new user_agent({ deviceCategory: 'desktop' });			
			await page.setUserAgent(userAgent.toString());			
			await page.setViewport({width: userAgent.data.screenWidth, height: userAgent.data.screenHeight});
			
			// scripts parsed by chrome
			const scriptsParsed: Array<any> = [];

			// note: add trailing slash since chrome adds it
			if (!url.endsWith('/'))
				url = url + '/';
										
			executionContexts.push('execution_createCDPSession');

			const client = await page.target().createCDPSession();
			await client.send('Debugger.enable');
			await client.send('Network.enable');     

			client.on('Debugger.scriptParsed', (params) => {

			try {
				scriptsParsed.push({'_scriptId':params.scriptId, '_scriptParsedData': params});                                    
			} catch {
                
             }

			});
        
            // urls hold redirect chain
		    let requestsChain : {[key: string]: any} = {};
		    const requestInterceptions: Array<{[key: string]: any}> = [];
            const responseInterceptions: Array<{[key: string]: any}> = [];	
        
			client.on('Network.requestWillBeSent', (e) => {
				
				const requestId = e.requestId;

				if(!requestsChain[requestId])
					requestsChain[requestId] = {};

				// check if url redirected
				if (typeof e.redirectResponse != 'undefined') {
					
					if(!requestsChain[requestId]['redirects'])
						requestsChain[requestId]['redirects'] = [];

						requestsChain[requestId]['redirects'].push({
						redirect: true,				
						status: e.redirectResponse.status,
						url: e.request.url,
						loader: e.loaderId,
						documentURL: e.documentURL,				
						type: e.type,
						frameId: e.frameId
						
					});

				} else {
				
					requestsChain[requestId] = {
							redirect: false,													
							url: e.request.url,							
							documentURL: e.documentURL,							
							type: e.type,
							frameId: e.frameId							
					};
				}
			});
		
			executionContexts.push('execution_setRequestInterception');

			const rootDomainRegex = utils.generateRootDomainRegex(url);
			const navigationRequestContext: Array<string> = [];	

			await page.setRequestInterception(true);	
							
			page.on('request', request => {
	
				try 
				{    													         								
					const url = request.url();        					
					
					if(!url.match(rootDomainRegex)) {     
						const resourceType = request.resourceType();
						const isNavigationRequest = request.isNavigationRequest();	 
						requestInterceptions.push({
							'navigationContext': _.last(navigationRequestContext),
							'context': _.last(executionContexts),
							'url': url,
							'resourceType': resourceType,
							'isNavigationRequest': isNavigationRequest,
							'headers': request.headers(),		
							'payload': request.postData()
												
						});
					}             
									
					request.continue();
	
				} catch (e) {}
	
			});
				
			page.on('response', async resp => {
				try 
				{    
					const url = resp.url();	
					const resourceType = resp.request().resourceType();	

					if(!url.match(rootDomainRegex)) {    

						let headers = undefined;	
						let payload = undefined;
																	
						try {							
							headers = await resp.headers();	
							if(!Consts.REFERRERS_CRAWLER.IGNORED_PAYLOADS_BY_RESOURCE_TYPES.includes(resourceType))
								payload = await resp.text();															
						} catch {}	

						responseInterceptions.push({
							'context': _.last(executionContexts),
							'url': url,
							'resourceType': resourceType, 
							'headers': headers,
							'payload': payload
						});
					}					
				}
				catch(e){}
			});
				
			let response = undefined;
			
			try {

				navigationRequestContext.push('navigationRequestContext_onload');
				executionContexts.push('execution_onload');
				    							
				try {
					response = await page.goto(url, {waitUntil: 'networkidle2'}); 										
				} catch(e) {					
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE_NETWORK_IDLE, message: e.message, stack: e.stack } as IBrowserError);
					requestsChain = {};
					scriptsParsed.splice(0, scriptsParsed.length);					
					requestInterceptions.splice(0, requestInterceptions.length);
					responseInterceptions.splice(0, responseInterceptions.length);					
				}
				finally {

					if(!response)
						response = await page.goto(url);
				}
							
				if(response)
				{
					pie_response['statusCode'] = response.status();
					pie_response['responseHeaders'] = response.headers();	
				}
						

			} catch (e) {		
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE, message: e.message, stack: e.stack } as IBrowserError);							
				throw(e);
			}

			try {
					
				executionContexts.push('execution_collectingHyperlinks');
				const hyperlinks = await page.$$eval('a', e=>e.map((a: HTMLAnchorElement)=>{
					return {
					 id: a.getAttribute('id'),
					 name: a.getAttribute('name'), 					  
					 className: a.getAttribute('class'),
					 href: a.getAttribute('href'),
					 innerText: a.innerText.replace(/^\s+|\s+$/gm,''),
					 rel: a.getAttribute('rel'),
					 referrerpolicy: a.getAttribute('referrerpolicy')
				   };
			   }));

			   if(hyperlinks && hyperlinks.length > 0)
			   		pie_response['hyperlinks'] = hyperlinks;
			  
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACTING_LINKS, message: e.message, stack: e.stack } as IBrowserError);			
			}
																			
			const searchInputValue = searchQuery;     	
			executionContexts.push('execution_makingElementsVisible');	
			try {

				await page.$eval(querySelector, (element, selector) => {
					
					try { 
						                           						                     
						let els = document.querySelectorAll(selector);						                                                   
						[].forEach.call(els, (el: HTMLElement ) => {       							                                                                                       
							el.style.display = 'block';
							el.style.visibility = 'visible';
							el.style.zIndex = '999999999999';                                       
							while (el.parentNode) {                            
								el = el.parentNode as HTMLElement;                                                         
								if (el.tagName === 'body')
								{
									return;
								}
								else
								{        
									if(el.style){
										el.style.display  = 'block';
										el.style.visibility = 'visible';
									}        
										
								} 
							}							
						});           
						
						return;
						
					} catch (e) {
						
					}																																																											
					}, `button[type="submit"], input[type*="submit"], ${querySelector}`);                    
				
			} catch (e) {				
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SET_ELEMENT_VISIBLE, message: e.message, stack: e.stack } as IBrowserError);	
			}

			try {

				executionContexts.push('execution_interactingWithInput');	
				await page.type(querySelector, searchInputValue, {delay: 500});  
				
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.TYPING_ON_ELEMENT, message: e.message, stack: e.stack } as IBrowserError);
			}
			
			try {
				
				executionContexts.push('execution_eventListenersAnalysis');	

				let context_listeners = await puppeteer_utils.performInputEventListenersAnalysis(client, querySelector, scriptsParsed);
				
				pie_response['listeners'].push({
					'context_listeners': context_listeners,
					'querySelector': querySelector,
					'inputElement': searchElement
				});       
							
			} catch (e) {				
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CAPTURING_EVENT_LISTENERS, message: e.message, stack: e.stack } as IBrowserError);				
			}
								
			try {     

				navigationRequestContext.push('navigationRequestContext_onsearch');
				executionContexts.push('execution_searchClick');	

				await Promise.all([
					page.type(querySelector, String.fromCharCode(13)),
					page.waitForNavigation({waitUntil: ['networkidle2', 'domcontentloaded']})
				]);
															
			} catch(e) {											
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.WAITING_FOR_SEARCH_RESULTS_NAVIGATION, message: e.message, stack: e.stack } as IBrowserError);
			}   
				
			try {

				executionContexts.push('execution_searchClickUrlExtraction');	

				pie_response['search_url'] = await Promise.race([page.evaluate(async ()=> { return window.location.href;}),
					page.waitFor(200000)
				]);
																
			} catch (e) {					
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACT_SEARCH_RESULTS_URL, message: e.message, stack: e.stack } as IBrowserError);
			}   
			
			finally {

				try {
					if(!pie_response['search_url'])					
						pie_response['search_url'] = page.url();										
				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACT_SEARCH_RESULTS_URL, message: e.message, stack: e.stack } as IBrowserError);
                }   
			}
																
			pie_response['requestsChain'] = requestsChain;									
			pie_response['requestInterceptions'] = requestInterceptions;
			pie_response['responseInterceptions'] = responseInterceptions;		
			
			try {						

				executionContexts.push('execution_closingPage');
				await page.close();					
						
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLOSING_PAGE, message: e.message, stack: e.stack } as IBrowserError);
			}
																	
			pie_response.status = { error: 0 };					

		} catch(e) {					
			pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLICK_STEP_EXCEPTION, message: e.message, stack: e.stack } as IBrowserError);
			pie_response.status = { error: 1, message: e.message, stack: e.stack, context: _.last(executionContexts) };			
		}
		finally {
			return resolve(pie_response);  
		}

	});
}

export const searchLookupAndClickSearchElement = async(browser: puppeteer.Browser, url: string, timeout: number, searchQuery: string, languageCodes: Array<string>) : Promise<any> => {

	return new Promise(async (resolve, reject) => {

		const pie_response = {	
			search_inputs_result: {
				 url: undefined,
				 searchInputs: [],			
				 status: undefined				
			} as SearchLookup,
			search_clicks_result: {
				querySelector: undefined,
				url: '',
				statusCode: -1,
				responseHeaders: undefined,				
				requestsChain: {},     
				listeners: [],
				requestInterceptions: [],
				responseInterceptions: [],
				hyperlinks: [],
				searchElement: undefined,         				
				search_url: null,
				status: undefined				
			} as ClickSearchElement,
			exceptions: [] as Array<IBrowserError>
		}  

		const timeoutId = setTimeout(() => {
            return reject();
        }, 300000);

		try {				
		
			const executionContexts: Array<string> = [];
			const navigationRequestContext: Array<string> = [];	
			const scriptsParsed: Array<any> = [];
			const requestInterceptions: Array<{[key: string]: any}> = [];
			const responseInterceptions: Array<{[key: string]: any}> = [];

			let requestsChain: {[key: string]: any} = {};
			let page = undefined; 
			let client = undefined;
			let searchElement = undefined;

			try 
			{			
				pie_response.search_inputs_result['url'] = url;
				
				page = await browser.newPage();
				
				// handling page errors.
				page.on('error', e => {
					throw e;
				});
				
				if(timeout) {
					page.setDefaultNavigationTimeout(timeout);
				}

				const userAgent = new user_agent({ deviceCategory: 'desktop' });			
				await page.setUserAgent(userAgent.toString());										
				await page.setViewport({width: userAgent.data.screenWidth, height: userAgent.data.screenHeight});
						
				// note: add trailing slash since chrome adds it
				if (!url.endsWith('/'))
					url = url + '/';
													
				executionContexts.push('execution_createCDPSession');

				client = await page.target().createCDPSession();
				await client.send('Debugger.enable');
				await client.send('Network.enable');     

				client.on('Debugger.scriptParsed', function(params) {

				try {
					scriptsParsed.push({'_scriptId':params.scriptId, '_scriptParsedData': params});                                    
				} catch (e) { }

				});
									
				client.on('Network.requestWillBeSent', (e) => {
												
					const requestId = e.requestId;

					if(!requestsChain[requestId])
						requestsChain[requestId] = {};
					
					if (typeof e.redirectResponse != 'undefined') {
						
						if(!requestsChain[requestId]['redirects'])
							requestsChain[requestId]['redirects'] = [];

							requestsChain[requestId]['redirects'].push({
							redirect: true,			
							status: e.redirectResponse.status,
							url: e.request.url,
							loader: e.loaderId,
							documentURL: e.documentURL,
							
							type: e.type,
							frameId: e.frameId
							
						});

					} else {
					
						requestsChain[requestId] = {
								redirect: false,							
								url: e.request.url,							
								documentURL: e.documentURL,							
								type: e.type,
								frameId: e.frameId													
						};
					}
				});
				
				executionContexts.push('execution_setRequestInterception');
				const rootDomainRegex = utils.generateRootDomainRegex(url);
				
				await page.setRequestInterception(true);	
								
				page.on('request', request => {
		
					try 
					{    													         								
						const url = request.url();        					
						if(!url.match(rootDomainRegex)) {     
							const resourceType = request.resourceType();
							const isNavigationRequest = request.isNavigationRequest();	 
							requestInterceptions.push({
								'navigationContext': _.last(navigationRequestContext),
								'context': _.last(executionContexts),
								'url': url,
								'resourceType': resourceType,
								'isNavigationRequest': isNavigationRequest,
								'headers': request.headers(),		
								'payload': request.postData()
													
							});
						}             
										
					request.continue();	
					} catch (e) {}
		
				});
					
				page.on('response', async resp => {
					try 
					{    
						const url = resp.url();	
						const resourceType = resp.request().resourceType();	

						if(!url.match(rootDomainRegex)) {    

							let headers = undefined;	
							let payload = undefined;
																		
							try {							
								headers = await resp.headers();	
								if(!Consts.REFERRERS_CRAWLER.IGNORED_PAYLOADS_BY_RESOURCE_TYPES.includes(resourceType))
									payload = await resp.text();															
							} catch {}	

							responseInterceptions.push({
								'context': _.last(executionContexts),
								'url': url,
								'resourceType': resourceType, 
								'headers': headers,
								'payload': payload
							});
						}					
					}
					catch(e){}
				});
						
				let response = undefined;

				try {

					navigationRequestContext.push('navigationRequestContext_onload');				
					executionContexts.push('execution_onload');

					try {
						response = await page.goto(url, {waitUntil: 'networkidle2'});
					} catch (e) {
						pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE_NETWORK_IDLE, message: e.message, stack: e.stack } as IBrowserError);
						requestsChain = {};
						scriptsParsed.splice(0, scriptsParsed.length);					
						requestInterceptions.splice(0, requestInterceptions.length);
						responseInterceptions.splice(0, responseInterceptions.length);																					
					}
					finally {
						if(!response)
							response = await page.goto(url);
					}
							
					if(response)
					{
						pie_response.search_clicks_result['statusCode'] = response.status();
						pie_response.search_clicks_result['responseHeaders'] = response.headers();		
					}	
							

				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE, message: e.message, stack: e.stack }  as IBrowserError);									
					throw(e);
				}

				// step 2
								
				executionContexts.push('execution_settingSearchInputs');				
				const searchRelatedElements: Array<string> = [];

				try {
					languageCodes.forEach(code => {
						const search_term: string = Consts.REFERRERS_CRAWLER.LANGUAGE_MAPPINGS.search_terms[code];
						if(search_term && search_term.length > 0)
							searchRelatedElements.push.apply(searchRelatedElements, Consts.REFERRERS_CRAWLER.SEARCH_QUERY_SELECTORS.map((d: string) => d.replace(/search/g, search_term)));
					});
				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SETUP_LANGUAGE_FOR_SEARCH_LOOKUPS, message: e.message, stack: e.stack } as IBrowserError);
					throw(e);				
				}
				
				executionContexts.push('execution_searchingInputs');

				for(let i = 0; i < searchRelatedElements.length ; i++ ) {             
					try 
					{               					                             
						const searchRelatedElementSelector = searchRelatedElements[i];					                             
								
						let inputs: Array<any> = [];
						
							if(searchRelatedElementSelector.startsWith('input'))
							{      
								try {
									
									inputs = await page.$$eval(searchRelatedElementSelector, e=>e.map((a)=>{
										return {
										id: a.getAttribute('id'),     
										type: a.getAttribute('type') || (a as any).type, 
										name: a.getAttribute('name'), 
										placeholder: a.getAttribute('placeholder'), 
										className: a.getAttribute('class')
									};
								}));
								
								} catch (e) {
									//TODO
								}         														
							}
							else
							{
								const searchRelatedElement = await page.$(searchRelatedElementSelector); 
										
								if(searchRelatedElement && searchRelatedElement.$$eval)
								{      
										
									try {
										inputs = await searchRelatedElement.$$eval('input', e=>e.map((el)=>{
											return {
											id: el.getAttribute('id'), 
											type: el.getAttribute('type') || (el as any).type, 
											name: el.getAttribute('name'), 
											placeholder: el.getAttribute('placeholder'), 
											className: el.getAttribute('class'),
											tagName: el.tagName.toLowerCase()
											};
										}));
									} catch (e) {
										pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack} as IBrowserError);
									}    													 																
								}
							}

							try {
										
								const element = await page.$eval(searchRelatedElementSelector, el => {
									return {
										id: el.getAttribute('id'), 
										type: el.getAttribute('type') || (el as any).type, 
										name: el.getAttribute('name'), 
										placeholder: el.getAttribute('placeholder'), 
										className: el.getAttribute('class'),
										tagName: el.tagName.toLowerCase()
									};
								});
								
								if(element && utils.validateSearchElementProps(element as HTMLInputElement))
								{						                                    				
									inputs.push(element);
								}
									

							} catch (e) {
								pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack} as IBrowserError);
							}  
																				
							if(inputs && inputs.length > 0)
							{                          
								const searchInputItem = {searchRelatedElementSelector, inputs};
								pie_response.search_inputs_result['searchInputs'].push(searchInputItem);                                                                                                                                
							}
							else
							{    
								try {
																											
								const remoteTryQuerySelector = searchRelatedElementSelector.startsWith('input') ? searchRelatedElementSelector : searchRelatedElementSelector + ' input';
								let remotely_fetched_inputs = await page.evaluate(async (selector_param) => {          
																
										let inputs = document.querySelectorAll(selector_param);
										let inputsNotSet: Array<any> = [];                                
										if(inputs && inputs.length > 0){
											[].forEach.call(inputs, (el: HTMLElement) => {                                                                               
												try {
													inputsNotSet.push({
														id: el.getAttribute('id'), 
														type: el.getAttribute('type') || (el as any).type, 
														name: el.getAttribute('name'), 
														placeholder: el.getAttribute('placeholder'), 
														className: el.getAttribute('class'),
														tagName: el.tagName.toLowerCase()                                 
												});          
												} catch (e) {
												
												}                                                                                
											});                         
										}                                                                
										return inputsNotSet;

								}, remoteTryQuerySelector);     
		
								if(remotely_fetched_inputs && remotely_fetched_inputs.length > 0)
								{                       
									const searchInputItem = {searchRelatedElementSelector, 'inputs': remotely_fetched_inputs};								
									pie_response.search_inputs_result['searchInputs'].push(searchInputItem);
								}
								} catch (e) {
									pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack} as IBrowserError);
								}     
							}
						
							if(pie_response.search_inputs_result['searchInputs'].length === 0)
							{
								try {
												
									const alternatives = await page.$$eval(`[href*="${utils.extractRootDomain(url)}"]`, e=>e.map((a: HTMLAnchorElement)=>{
										return {
											id: a.getAttribute('id'),
											name: a.getAttribute('name'), 					  
											className: a.getAttribute('class'),
											href: a.getAttribute('href'),
											innerText: a.innerText.replace(/^\s+|\s+$/gm,''),
											rel: a.getAttribute('rel'),
											referrerpolicy: a.getAttribute('referrerpolicy')				  
									};
								}));

								if(alternatives && alternatives.length > 0)
									pie_response.search_inputs_result['alternatives'] = alternatives;
								
								} catch (e) {
									pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack} as IBrowserError);
								}         		
							}

																	
					} catch(e) {    					
						pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_LOOKUP_LOOP, message: e.message, stack: e.stack} as IBrowserError);
					}           
				}
									
				pie_response.search_inputs_result.status = { error: 0 };				
				
			} catch (e) { 	
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.LOOKUP_STEP_EXCEPTION, message: e.message, stack: e.stack} as IBrowserError);
				pie_response.search_inputs_result.status = { error: 1, message: e.message, stack: e.stack, context: _.last(executionContexts) };		      						
				throw(e);						
			}

			try {

				executionContexts.push('extractingSearchInput');

				let searchInputs = pie_response.search_inputs_result['searchInputs'];
				let usedSelectors = []; 
				for(let i = 0; i< searchInputs.length; i++) {
					let relevant_search_inputs = searchInputs[i].inputs.filter((d) => d.type == 'text' || d.type == 'search');            
					usedSelectors.push(...relevant_search_inputs);    
				}
				usedSelectors = _.uniqBy(usedSelectors, (input: HTMLInputElement) => {
					return utils.buildQuerySelector(input,'input');
				});
				
				if(usedSelectors && usedSelectors.length > 0)
				{
					searchElement = usedSelectors[0];
				}
				else
				{
					pie_response.search_clicks_result = { 'error': 0, 'noinputs': 1 } as IClickSearchElementError;
					return resolve(pie_response);
				}
				
			} catch (e) {	
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.PARSING_QUERY_SELECTOR, message: e.message, stack: e.stack} as IBrowserError);				
				throw(e);
			}

			// step 3
			executionContexts.length = 0;		
			executionContexts.push('execution_prepQuerySelector');
			
			try {
						
				pie_response.search_clicks_result['url'] = url;
				pie_response.search_clicks_result['searchElement'] = searchElement;

				const querySelector = pie_response.search_clicks_result['querySelector'] =  utils.buildQuerySelector(searchElement,'input');
				
				try {
						
					executionContexts.push('execution_collectingHyperlinks');
					const hyperlinks = await page.$$eval('a', e=>e.map((a: HTMLAnchorElement)=>{
						return {
						id: a.getAttribute('id'),
						name: a.getAttribute('name'), 					  
						className: a.getAttribute('class'),
						href: a.getAttribute('href'),
						innerText: a.innerText.replace(/^\s+|\s+$/gm,''),
						rel: a.getAttribute('rel'),
						referrerpolicy: a.getAttribute('referrerpolicy')
					};
					}));

					if(hyperlinks && hyperlinks.length > 0)
						pie_response.search_clicks_result['hyperlinks'] = hyperlinks;
				
				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACTING_LINKS, message: e.message, stack: e.stack } as IBrowserError);			
				}
																																														
				const searchInputValue = searchQuery;     	
				executionContexts.push('execution_makingElementsVisible');				
				try {

					await page.$eval(querySelector, (element, selector) => {
						
						try {      

							let els = document.querySelectorAll(selector);                                                   
							[].forEach.call(els, (el: HTMLElement) => {                                                                                   
								el.style.display = 'block';
								el.style.visibility = 'visible';
								el.style.zIndex = '999999999999';                                    
								while (el.parentNode) {                            
									el = el.parentNode as HTMLElement;                                                         
									if (el.tagName === 'body')
									{
										return;
									}
									else
									{        
										if(el.style){
											el.style.display  = 'block';
											el.style.visibility = 'visible';
										}     
									} 
								}							
							});           
							
							return;
							
						} catch (e) {
							
						}																																																											
						}, `button[type="submit"], input[type*="submit"], ${querySelector}`);                    
					
				} catch (e) {				
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SET_ELEMENT_VISIBLE, message: e.message, stack: e.stack } as IBrowserError);
				}
				
				try {

					executionContexts.push('execution_interactingWithInput');	
					await page.type(querySelector, searchInputValue, {delay: 500}); 
					
				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.TYPING_ON_ELEMENT, message: e.message, stack: e.stack } as IBrowserError);
				}

				try {
					
					executionContexts.push('execution_eventListenersAnalysis');	
					let context_listeners = await puppeteer_utils.performInputEventListenersAnalysis(client, querySelector, scriptsParsed);
					
					pie_response.search_clicks_result['listeners'].push({
						'context_listeners': context_listeners,
						'querySelector': querySelector,
						'inputElement': searchElement
					});       				
				
				} catch (e) {				
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CAPTURING_EVENT_LISTENERS, message: e.message, stack: e.stack } as IBrowserError);
				}
						
				try {    
								
					navigationRequestContext.push('navigationRequestContext_onsearch');
					executionContexts.push('execution_searchClick');	

					await Promise.all([
						page.type(querySelector, String.fromCharCode(13)),
						page.waitForNavigation({waitUntil: ['networkidle2', 'domcontentloaded']})
					]);
																
				} catch(e) {											
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.WAITING_FOR_SEARCH_RESULTS_NAVIGATION, message: e.message, stack: e.stack } as IBrowserError);
				}   
					
				try {

					executionContexts.push('execution_searchClickUrlExtraction');	
	
					pie_response.search_clicks_result['search_url'] = await Promise.race([page.evaluate(async ()=> { return window.location.href;}),
						page.waitFor(200000)
					]);
																	
				} catch (e) {					
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACT_SEARCH_RESULTS_URL, message: e.message, stack: e.stack } as IBrowserError);
				}
				finally {
	
					try {
						if(!pie_response.search_clicks_result['search_url'])					
						pie_response.search_clicks_result['search_url'] = page.url();										
					} catch (e) {
						pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACT_SEARCH_RESULTS_URL, message: e.message, stack: e.stack } as IBrowserError);
					}
	
				}
				
				pie_response.search_clicks_result['requestsChain'] = requestsChain;									
				pie_response.search_clicks_result['requestInterceptions'] = requestInterceptions;
				pie_response.search_clicks_result['responseInterceptions'] = responseInterceptions;		

				
				try {						

					executionContexts.push('execution_closingPage');
					await page.close();	
																				
				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLOSING_PAGE, message: e.message, stack: e.stack } as IBrowserError);
				}
																		
				pie_response.search_clicks_result.status = { error: 0 };		
				return resolve(pie_response);  

			} catch(e) {
				pie_response.search_clicks_result.status = { error: 1, message: e.message, stack: e.stack, context: _.last(executionContexts) };									
				throw(e);
			}

		} catch (e) {
			pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.LOOKUP_AND_CLICK_STEP_EXCEPTION, message: e.message, stack: e.stack} as IBrowserError);			
		}
		finally {			
			return resolve(pie_response);
		}

	});
}

export const searchBySearchEndpointUrl = async(browser: puppeteer.Browser, url: string, timeout: number) : Promise<ClickSearchElement> => {

	return new Promise(async (resolve, reject) => {

		const pie_response = {					
			querySelector: null,
			url: '',
			statusCode: -1,
			responseHeaders: undefined,				
			requestsChain: {},     
			listeners: [],
			requestInterceptions: [],
			responseInterceptions: [],
			hyperlinks: [],
			searchElement: null,         				
			search_url: null,
			status: undefined
		}  as ClickSearchElement

		const timeoutId = setTimeout(() => {
            return reject();
        }, 300000);

		const executionContexts = ['execution_onload'];
		
		try 		
		{								 
			pie_response['url'] = pie_response['search_url'] = url;
			
			const rootDomainRegex = utils.generateRootDomainRegex(url);		
			const page = await browser.newPage();
			
			// handling page errors.
			page.on('error', e => {
				throw e;
			});
	
			if(timeout) {		
				page.setDefaultNavigationTimeout(timeout);
			}
			
			const userAgent = new user_agent({ deviceCategory: 'desktop' });			
			await page.setUserAgent(userAgent.toString());			
			await page.setViewport({width: userAgent.data.screenWidth, height: userAgent.data.screenHeight});
			
			// note: add trailing slash since chrome adds it
			if (!url.endsWith('/'))
				url = url + '/';
	
			// urls hold redirect chain
			let requestsChain: {[key: string]: any} = {};
			const requestInterceptions: Array<{[key: string]: any}> = [];
			const responseInterceptions: Array<{[key: string]: any}> = [];
	
			const client = await page.target().createCDPSession();
			await client.send('Debugger.enable');	
			await client.send('Network.enable');
			
			client.on('Network.requestWillBeSent', (e) => {
				
				const requestId = e.requestId;

				if(!requestsChain[requestId])
					requestsChain[requestId] = {};

				// check if url redirected
				if (typeof e.redirectResponse != 'undefined') {
					
					if(!requestsChain[requestId]['redirects'])
						requestsChain[requestId]['redirects'] = [];

						requestsChain[requestId]['redirects'].push({
						redirect: true,				
						status: e.redirectResponse.status,
						url: e.request.url,
						loader: e.loaderId,
						documentURL: e.documentURL,					
						type: e.type,
						frameId: e.frameId
						
					});

				} else {
				
					requestsChain[requestId] = {
							redirect: false,													
							url: e.request.url,							
							documentURL: e.documentURL,			
							type: e.type,
							frameId: e.frameId												
					};
				}
			});
		
			await page.setRequestInterception(true);
					
			page.on('request', request => {
	
				try 
				{    
					const url = request.url();        
					const resourceType = request.resourceType();  
					if(!url.match(rootDomainRegex)) {   		        								
					requestInterceptions.push({
							'context': _.last(executionContexts),
							'url': url,
							'headers': request.headers(),
							'method': request.method(),
							'resourceType': resourceType,							
							'payload': request.postData()
							
						});
					}             
									
				request.continue();
	
				} catch (e) {					
				}
	
			});
				
			page.on('response', async resp => {

				try 
				{    
					const url = resp.url();	
					
					if(!url.match(rootDomainRegex)) {    
						
						let resourceType = resp.request().resourceType();
						let method = resp.request().method();				
						let headers = undefined;	
						let payload = undefined;	
						
						try {

							headers = await resp.headers();	
							if(!Consts.REFERRERS_CRAWLER.IGNORED_PAYLOADS_BY_RESOURCE_TYPES.includes(resourceType))
							{
								payload = await resp.text();														
							}
							else
							{
								payload = null;
							}
								
						} catch {}	

						responseInterceptions.push({
							'context': _.last(executionContexts),
							'url': url,						
							'headers': headers,
							'method': method,
							'resourceType': resourceType, 														
							'payload': payload										
						});
					}					
				}
				catch(e){						
				}
			});
				
			let response = undefined;
						
			try {	

					try {
						response = await page.goto(url, {waitUntil: 'networkidle2'}); 									
					} catch (e) {
						pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE_NETWORK_IDLE, message: e.message, stack: e.stack } as IBrowserError);														
						requestsChain = {};
						requestInterceptions.splice(0, requestInterceptions.length);
						responseInterceptions.splice(0, responseInterceptions.length);						
					}
					finally {

						if(!response)
							response = await page.goto(url);
						
					}

					if(response)
					{
						pie_response['statusCode'] = response.status();
						pie_response['responseHeaders'] = response.headers();
					}
																						
					pie_response['requestsChain'] = requestsChain;									
					pie_response['requestInterceptions'] = requestInterceptions;
					pie_response['responseInterceptions'] = responseInterceptions;							
										
			} catch (e) {
				
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE, message: e.message, stack: e.stack } as IBrowserError);		
				throw(e);
			}
			
			try {
					
				executionContexts.push('execution_collectingHyperlinks');

				const hyperlinks = await page.$$eval('a', e=>e.map((a: HTMLAnchorElement)=>{
					return {
					 id: a.getAttribute('id'),
					 name: a.getAttribute('name'), 					  
					 className: a.getAttribute('class'),
					 href: a.getAttribute('href'),
					 innerText: a.innerText.replace(/^\s+|\s+$/gm,''),
					 rel: a.getAttribute('rel'),
					 referrerpolicy: a.getAttribute('referrerpolicy')
				   };
			   }));

			   if(hyperlinks && hyperlinks.length > 0)
			   		pie_response['hyperlinks'] = hyperlinks;
			  
			} catch (e) {				
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACTING_LINKS, message: e.message, stack: e.stack } as IBrowserError);
			}
											
			try {						

				executionContexts.push('execution_closingPage');	
				await page.close();				
								
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLOSING_PAGE, message: e.message, stack: e.stack } as IBrowserError);
			}	
			         					
			pie_response.status = { error: 0 };					
			
		} catch (e) { 		
			pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SEARCH_ENDPOINT_URL_EXCEPTION, message: e.message, stack: e.stack} as IBrowserError);		
			pie_response.status = { error: 1, message: e.message, stack: e.stack, context: _.last(executionContexts) };			
		}
		finally {
			return resolve(pie_response);  
		}

	});
}

// NEW STEP TASKS

const collectUsefulSearchCollectors = async (page: puppeteer.Page, querySelector: string): Promise<any> => {

	return new Promise(async (resolve, reject) => {
	
		try {

			let remotely_fetched_inputs = await page.evaluate(async (selector_param: string) => {          
																
				let inputs = document.querySelectorAll(selector_param);
				let inputsNotSet: Array<any> = [];                                
				if(inputs && inputs.length > 0){
					[].forEach.call(inputs, function(el: HTMLElement) {                                                                               
						try {
							inputsNotSet.push({
								id: el.getAttribute('id'), 
								type: el.getAttribute('type') || (el as any).type, 
								name: el.getAttribute('name'), 
								placeholder: el.getAttribute('placeholder'), 
								className: el.getAttribute('class'),
								tagName: el.tagName.toLowerCase()												
						});          
						} catch (e) {
						
						}                                                                                
					});                         
				}                                                                
				return inputsNotSet;

		}, querySelector);     

		if(!remotely_fetched_inputs || !remotely_fetched_inputs.length)
			return resolve(null);		

		const validInputSelectors = _.uniqBy(remotely_fetched_inputs, (input: HTMLInputElement) => {
			return utils.buildQuerySelector(input,'input');
		});

		if(validInputSelectors && validInputSelectors.length > 0)
			return resolve(validInputSelectors[0]); 

		return resolve(undefined);

		} catch (e) {
			return reject(e);
		}
		
	});
}

export const clickAndWaitForSearchElement = async(url: string, searchElement: any, timeout: number, searchQuery: string) : Promise<ClickSearchElement> => {

	return new Promise(async (resolve, reject) => {

		const pie_response = {					
			querySelector: undefined,
			url: '',
			statusCode: -1,
			responseHeaders: undefined,				
			requestsChain: {},     
			listeners: [],
			requestInterceptions: [],
			responseInterceptions: [],
			hyperlinks: [],
			searchElement: undefined,         				
			search_url: null,
			status: undefined,
			exceptions: []
		}  as ClickSearchElement

		const timeoutId = setTimeout(() => {
            return reject();
        }, 300000);

        const executionContexts = ['execution_prepQuerySelector'];        
       
		try {
				
			const browser = await BrowserUtils.getBrowser({
				headless: true,
				args: ['--enable-features=NetworkService', '--no-sandbox=true', '--window-size=1900,1050', '--disable-notifications',
				'--ignore-certificate-errors', '--disable-dev-shm-usage=true', '--disable-setuid-sandbox=true', '--disable-accelerated-2d-canvas=true', '--disable-gpu=true'],
				ignoreHTTPSErrors: true
			}); 
			pie_response['url'] = url;
			pie_response['searchElement'] = searchElement;

			const querySelector = pie_response['querySelector'] =  utils.buildGenericQuerySelector(searchElement, searchElement.tagName); //utils.buildQuerySelector(searchElement,'input');																														

			const page = await browser.newPage();
			
			// handling page errors.
			// page.on('error', e => {
			
			// });
					
			if(timeout) {					
				page.setDefaultNavigationTimeout(timeout);
			}
			
			const userAgent = new user_agent({ deviceCategory: 'desktop' });			
			await page.setUserAgent(userAgent.toString());			
			await page.setViewport({width: userAgent.data.screenWidth, height: userAgent.data.screenHeight});
			
			// scripts parsed by chrome
			const scriptsParsed: Array<any> = [];

			// note: add trailing slash since chrome adds it
			if (!url.endsWith('/'))
				url = url + '/';
										
			executionContexts.push('execution_createCDPSession');

			const client = await page.target().createCDPSession();
			await client.send('Debugger.enable');
			await client.send('Network.enable');     

			client.on('Debugger.scriptParsed', (params) => {

			try {
				scriptsParsed.push({'_scriptId':params.scriptId, '_scriptParsedData': params});                                    
			} catch {
                
             }

			});
        
            // urls hold redirect chain
		    let requestsChain : {[key: string]: any} = {};
		    const requestInterceptions: Array<{[key: string]: any}> = [];
            const responseInterceptions: Array<{[key: string]: any}> = [];	
        
			client.on('Network.requestWillBeSent', (e) => {
				
				const requestId = e.requestId;

				if(!requestsChain[requestId])
					requestsChain[requestId] = {};

				// check if url redirected
				if (typeof e.redirectResponse != 'undefined') {
					
					if(!requestsChain[requestId]['redirects'])
						requestsChain[requestId]['redirects'] = [];

						requestsChain[requestId]['redirects'].push({
						redirect: true,				
						status: e.redirectResponse.status,
						url: e.request.url,
						loader: e.loaderId,
						documentURL: e.documentURL,				
						type: e.type,
						frameId: e.frameId
						
					});

				} else {
				
					requestsChain[requestId] = {
							redirect: false,													
							url: e.request.url,							
							documentURL: e.documentURL,							
							type: e.type,
							frameId: e.frameId							
					};
				}
			});
		
			executionContexts.push('execution_setRequestInterception');

			const rootDomainRegex = utils.generateRootDomainRegex(url);
			const navigationRequestContext: Array<string> = [];	

			await page.setRequestInterception(true);	
							
			page.on('request', request => {
	
				try 
				{    													         								
					const url = request.url();        					
					
					if(!url.match(rootDomainRegex)) {     
						const resourceType = request.resourceType();
						const isNavigationRequest = request.isNavigationRequest();	 
						requestInterceptions.push({
							'navigationContext': _.last(navigationRequestContext),
							'context': _.last(executionContexts),
							'url': url,
							'resourceType': resourceType,
							'isNavigationRequest': isNavigationRequest,
							'headers': request.headers(),		
							'payload': request.postData()
												
						});
					}             
									
					request.continue();
	
				} catch (e) {}
	
			});
				
			page.on('response', async resp => {
				try 
				{    
					const url = resp.url();	
					const resourceType = resp.request().resourceType();	

					if(!url.match(rootDomainRegex)) {    

						let headers = undefined;	
						let payload = undefined;
																	
						try {							
							headers = await resp.headers();	
							if(!Consts.REFERRERS_CRAWLER.IGNORED_PAYLOADS_BY_RESOURCE_TYPES.includes(resourceType))
								payload = await resp.text();															
						} catch {}	

						responseInterceptions.push({
							'context': _.last(executionContexts),
							'url': url,
							'resourceType': resourceType, 
							'headers': headers,
							'payload': payload
						});
					}					
				}
				catch(e){}
			});
				
			let response = undefined;
			
			try {

				navigationRequestContext.push('navigationRequestContext_onload');
				executionContexts.push('execution_onload');
				    							
				try {
					response = await page.goto(url, {waitUntil: 'networkidle2'}); 										
				} catch(e) {					
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE_NETWORK_IDLE, message: e.message, stack: e.stack } as IBrowserError);
					requestsChain = {};
					scriptsParsed.splice(0, scriptsParsed.length);					
					requestInterceptions.splice(0, requestInterceptions.length);
					responseInterceptions.splice(0, responseInterceptions.length);					
				}
				finally {

					if(!response)
						response = await page.goto(url);
				}
							
				if(response)
				{
					pie_response['statusCode'] = response.status();
					pie_response['responseHeaders'] = response.headers();	
				}
						

			} catch (e) {		
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE, message: e.message, stack: e.stack } as IBrowserError);							
				throw(e);
			}

			await page.click(querySelector);
			await page.waitFor(10000);

			const relevantSelectors = `input[type*="search" i], input[class*="search" i], div[class*="search" i] input`;
			const searchElementToClick = await collectUsefulSearchCollectors(page, relevantSelectors);

			if(searchElementToClick && searchElementToClick !== null)
			{
				const updated_inputSearch_QuerySelector = utils.buildGenericQuerySelector(searchElementToClick, searchElementToClick.tagName);
				pie_response['searchElement'] = searchElementToClick;
				pie_response['querySelector'] = updated_inputSearch_QuerySelector;
				
				try {

					requestsChain = {};
					scriptsParsed.splice(0, scriptsParsed.length);					
					requestInterceptions.splice(0, requestInterceptions.length);
					responseInterceptions.splice(0, responseInterceptions.length);		
					
					executionContexts.push('execution_collectingHyperlinks');
					const hyperlinks = await page.$$eval('a', e=>e.map((a: HTMLAnchorElement)=>{
						return {
						 id: a.getAttribute('id'),
						 name: a.getAttribute('name'), 					  
						 className: a.getAttribute('class'),
						 href: a.getAttribute('href'),
						 innerText: a.innerText.replace(/^\s+|\s+$/gm,''),
						 rel: a.getAttribute('rel'),
						 referrerpolicy: a.getAttribute('referrerpolicy')
					   };
				   }));
	
				   if(hyperlinks && hyperlinks.length > 0)
						   pie_response['hyperlinks'] = hyperlinks;
				  
				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACTING_LINKS, message: e.message, stack: e.stack } as IBrowserError);			
				}
																				
				const searchInputValue = searchQuery;     	
				executionContexts.push('execution_makingElementsVisible');	
				try {
	
					await page.$eval(querySelector, (element, selector) => {
						
						try { 
																									
							let els = document.querySelectorAll(selector);						                                                   
							[].forEach.call(els, (el: HTMLElement ) => {       							                                                                                       
								el.style.display = 'block';
								el.style.visibility = 'visible';
								el.style.zIndex = '999999999999';                                       
								while (el.parentNode) {                            
									el = el.parentNode as HTMLElement;                                                         
									if (el.tagName === 'body')
									{
										return;
									}
									else
									{        
										if(el.style){
											el.style.display  = 'block';
											el.style.visibility = 'visible';
										}        
											
									} 
								}							
							});           
							
							return;
							
						} catch (e) {
							
						}																																																											
						}, `button[type="submit"], input[type*="submit"], ${querySelector}`);                    
					
				} catch (e) {				
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SET_ELEMENT_VISIBLE, message: e.message, stack: e.stack } as IBrowserError);	
				}
	
				try {
	
					executionContexts.push('execution_interactingWithInput');	
					await page.type(pie_response['querySelector'], searchInputValue, {delay: 500});  
					
				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.TYPING_ON_ELEMENT, message: e.message, stack: e.stack } as IBrowserError);
				}
				
				try {
					
					executionContexts.push('execution_eventListenersAnalysis');	
	
					let context_listeners = await puppeteer_utils.performInputEventListenersAnalysis(client, pie_response['querySelector'], scriptsParsed);
					
					pie_response['listeners'].push({
						'context_listeners': context_listeners,
						'querySelector': querySelector,
						'inputElement': searchElement
					});       
								
				} catch (e) {				
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CAPTURING_EVENT_LISTENERS, message: e.message, stack: e.stack } as IBrowserError);				
				}
									
				try {     
	
					navigationRequestContext.push('navigationRequestContext_onsearch');
					executionContexts.push('execution_searchClick');	
	
					await Promise.all([
						page.type(pie_response['querySelector'], String.fromCharCode(13)),
						page.waitForNavigation({waitUntil: ['networkidle2', 'domcontentloaded']})
					]);
																
				} catch(e) {											
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.WAITING_FOR_SEARCH_RESULTS_NAVIGATION, message: e.message, stack: e.stack } as IBrowserError);
				}   
					
				try {
	
					executionContexts.push('execution_searchClickUrlExtraction');	
	
					pie_response['search_url'] = await Promise.race([page.evaluate(async ()=> { return window.location.href;}),
						page.waitFor(200000)
					]);
																	
				} catch (e) {					
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACT_SEARCH_RESULTS_URL, message: e.message, stack: e.stack } as IBrowserError);
				}   
				
				finally {
	
					try {
						if(!pie_response['search_url'])					
							pie_response['search_url'] = page.url();										
					} catch (e) {
						pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACT_SEARCH_RESULTS_URL, message: e.message, stack: e.stack } as IBrowserError);
					}   
				}
																	
				pie_response['requestsChain'] = requestsChain;									
				pie_response['requestInterceptions'] = requestInterceptions;
				pie_response['responseInterceptions'] = responseInterceptions;		
				
				try {						
	
					executionContexts.push('execution_closingPage');
					await page.close();					
							
				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLOSING_PAGE, message: e.message, stack: e.stack } as IBrowserError);
				}
																		
				pie_response.status = { error: 0 };	
			}

			await BrowserUtils.closeOpenPages(browser);
			await browser.close();								

		} catch(e) {					
			pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLICK_STEP_EXCEPTION, message: e.message, stack: e.stack } as IBrowserError);
			pie_response.status = { error: 1, message: e.message, stack: e.stack, context: _.last(executionContexts) };			
		}
		finally {
			return resolve(pie_response);  
		}

	});
}

const clickAndDetectNewTab = async (page: puppeteer.Page, browser: puppeteer.Browser, selector: string) : Promise<any> => {

	//save target of original page to know that this was the opener:     
	const pageTarget = page.target();
	//execute click on first tab that triggers opening of new tab:
	await page.click(selector);
	//check that the first page opened this new page:
	const newTarget = await browser.waitForTarget(target => target.opener() === pageTarget);
	//get the new page object:
	const newPage = await newTarget.page();

	return undefined;

}

export const clickSearchElementAnotherTab = async(browser: puppeteer.Browser, url: string, searchElement: any, timeout: number, searchQuery: string) : Promise<ClickSearchElement> => {

	return new Promise(async (resolve, reject) => {

		let pie_response = {					
			querySelector: undefined,
			url: '',
			statusCode: -1,
			responseHeaders: undefined,				
			requestsChain: {},     
			listeners: [],
			requestInterceptions: [],
			responseInterceptions: [],
			hyperlinks: [],
			searchElement: undefined,         				
			search_url: null,
			status: undefined,
			exceptions: []
		}  as ClickSearchElement

		const timeoutId = setTimeout(() => {
            return reject();
        }, 300000);

        const executionContexts = ['execution_prepQuerySelector'];
        let page: puppeteer.Page;
       
		try {
				
			pie_response['url'] = url;
			pie_response['searchElement'] = searchElement;

			const querySelector = pie_response['querySelector'] =  utils.buildQuerySelector(searchElement,'input');																														

			page = await browser.newPage();
			
			// handling page errors.
			page.on('error', e => {
				throw e;
			});
					
			if(timeout) {					
				page.setDefaultNavigationTimeout(timeout);
			}
			
			const userAgent = new user_agent({ deviceCategory: 'desktop' });			
			await page.setUserAgent(userAgent.toString());			
			await page.setViewport({width: userAgent.data.screenWidth, height: userAgent.data.screenHeight});
			
			// scripts parsed by chrome
			const scriptsParsed: Array<any> = [];

			// note: add trailing slash since chrome adds it
			if (!url.endsWith('/'))
				url = url + '/';
										
			executionContexts.push('execution_createCDPSession');

			const client = await page.target().createCDPSession();
			await client.send('Debugger.enable');
			await client.send('Network.enable');     

			client.on('Debugger.scriptParsed', (params) => {

			try {
				scriptsParsed.push({'_scriptId':params.scriptId, '_scriptParsedData': params});                                    
			} catch {
                
             }

			});
        
            // urls hold redirect chain
		    let requestsChain : {[key: string]: any} = {};
		    const requestInterceptions: Array<{[key: string]: any}> = [];
            const responseInterceptions: Array<{[key: string]: any}> = [];	
        
			client.on('Network.requestWillBeSent', (e) => {
				
				const requestId = e.requestId;

				if(!requestsChain[requestId])
					requestsChain[requestId] = {};

				// check if url redirected
				if (typeof e.redirectResponse != 'undefined') {
					
					if(!requestsChain[requestId]['redirects'])
						requestsChain[requestId]['redirects'] = [];

						requestsChain[requestId]['redirects'].push({
						redirect: true,				
						status: e.redirectResponse.status,
						url: e.request.url,
						loader: e.loaderId,
						documentURL: e.documentURL,				
						type: e.type,
						frameId: e.frameId
						
					});

				} else {
				
					requestsChain[requestId] = {
							redirect: false,													
							url: e.request.url,							
							documentURL: e.documentURL,							
							type: e.type,
							frameId: e.frameId							
					};
				}
			});
		
			executionContexts.push('execution_setRequestInterception');

			const rootDomainRegex = utils.generateRootDomainRegex(url);
			const navigationRequestContext: Array<string> = [];	

			await page.setRequestInterception(true);	
							
			page.on('request', request => {
	
				try 
				{    													         								
					const url = request.url();        					
					
					if(!url.match(rootDomainRegex)) {     
						const resourceType = request.resourceType();
						const isNavigationRequest = request.isNavigationRequest();	 
						requestInterceptions.push({
							'navigationContext': _.last(navigationRequestContext),
							'context': _.last(executionContexts),
							'url': url,
							'resourceType': resourceType,
							'isNavigationRequest': isNavigationRequest,
							'headers': request.headers(),		
							'payload': request.postData()
												
						});
					}             
									
					request.continue();
	
				} catch (e) {}
	
			});
				
			page.on('response', async resp => {
				try 
				{    
					const url = resp.url();	
					const resourceType = resp.request().resourceType();	

					if(!url.match(rootDomainRegex)) {    

						let headers = undefined;	
						let payload = undefined;
																	
						try {							
							headers = await resp.headers();	
							if(!Consts.REFERRERS_CRAWLER.IGNORED_PAYLOADS_BY_RESOURCE_TYPES.includes(resourceType))
								payload = await resp.text();															
						} catch {}	

						responseInterceptions.push({
							'context': _.last(executionContexts),
							'url': url,
							'resourceType': resourceType, 
							'headers': headers,
							'payload': payload
						});
					}					
				}
				catch(e){}
			});
				
			let response = undefined;
			
			try {

				navigationRequestContext.push('navigationRequestContext_onload');
				executionContexts.push('execution_onload');
				    							
				try {
					response = await page.goto(url, {waitUntil: 'networkidle2'}); 										
				} catch(e) {					
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE_NETWORK_IDLE, message: e.message, stack: e.stack } as IBrowserError);
					requestsChain = {};
					scriptsParsed.splice(0, scriptsParsed.length);					
					requestInterceptions.splice(0, requestInterceptions.length);
					responseInterceptions.splice(0, responseInterceptions.length);					
				}
				finally {

					if(!response)
						response = await page.goto(url);
				}
							
				if(response)
				{
					pie_response['statusCode'] = response.status();
					pie_response['responseHeaders'] = response.headers();	
				}
						

			} catch (e) {		
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE, message: e.message, stack: e.stack } as IBrowserError);							
				throw(e);
			}

			try {
					
				executionContexts.push('execution_collectingHyperlinks');
				const hyperlinks = await page.$$eval('a', e=>e.map((a: HTMLAnchorElement)=>{
					return {
					 id: a.getAttribute('id'),
					 name: a.getAttribute('name'), 					  
					 className: a.getAttribute('class'),
					 href: a.getAttribute('href'),
					 innerText: a.innerText.replace(/^\s+|\s+$/gm,''),
					 rel: a.getAttribute('rel'),
					 referrerpolicy: a.getAttribute('referrerpolicy')
				   };
			   }));

			   if(hyperlinks && hyperlinks.length > 0)
			   		pie_response['hyperlinks'] = hyperlinks;
			  
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACTING_LINKS, message: e.message, stack: e.stack } as IBrowserError);			
			}
																			
			const searchInputValue = searchQuery;     	
			executionContexts.push('execution_makingElementsVisible');	
			try {

				await page.$eval(querySelector, (element, selector) => {
					
					try { 
						                           						                     
						let els = document.querySelectorAll(selector);						                                                   
						[].forEach.call(els, (el: HTMLElement ) => {       							                                                                                       
							el.style.display = 'block';
							el.style.visibility = 'visible';
							el.style.zIndex = '999999999999';                                       
							while (el.parentNode) {                            
								el = el.parentNode as HTMLElement;                                                         
								if (el.tagName === 'body')
								{
									return;
								}
								else
								{        
									if(el.style){
										el.style.display  = 'block';
										el.style.visibility = 'visible';
									}        
										
								} 
							}							
						});           
						
						return;
						
					} catch (e) {
						
					}																																																											
					}, `button[type="submit"], input[type*="submit"], ${querySelector}`);                    
				
			} catch (e) {				
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SET_ELEMENT_VISIBLE, message: e.message, stack: e.stack } as IBrowserError);	
			}

			try {

				executionContexts.push('execution_interactingWithInput');	
				await page.type(querySelector, searchInputValue, {delay: 500});  
				
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.TYPING_ON_ELEMENT, message: e.message, stack: e.stack } as IBrowserError);
			}
			
			try {
				
				executionContexts.push('execution_eventListenersAnalysis');	

				let context_listeners = await puppeteer_utils.performInputEventListenersAnalysis(client, querySelector, scriptsParsed);
				
				pie_response['listeners'].push({
					'context_listeners': context_listeners,
					'querySelector': querySelector,
					'inputElement': searchElement
				});       
							
			} catch (e) {				
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CAPTURING_EVENT_LISTENERS, message: e.message, stack: e.stack } as IBrowserError);				
			}
								
			try {     

				navigationRequestContext.push('navigationRequestContext_onsearch');
				executionContexts.push('execution_searchClick');					
				await page.type(querySelector, String.fromCharCode(13));									

				//save target of original page to know that this was the opener:     
				const pageTarget = page.target();
				//check that the first page opened this new page:
				const newTarget = await browser.waitForTarget(target => target.opener() === pageTarget);
				//get the new page object:
				const newPage = await newTarget.page();
				// get new page URL.
				const newPageURL = newPage.url();
				if(newPageURL && newPageURL.length > 0)
				{
					pie_response = await searchBySearchEndpointUrl(browser, newPageURL, timeout);
					return resolve(pie_response);
				}
															
			} catch(e) {											
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.WAITING_FOR_SEARCH_RESULTS_NAVIGATION, message: e.message, stack: e.stack } as IBrowserError);
			}   
		
		
			try {						

				executionContexts.push('execution_closingPage');
				await page.close();					
						
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLOSING_PAGE, message: e.message, stack: e.stack } as IBrowserError);
			}
																							

		} catch(e) {					
			pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLICK_STEP_EXCEPTION, message: e.message, stack: e.stack } as IBrowserError);
			pie_response.status = { error: 1, message: e.message, stack: e.stack, context: _.last(executionContexts) };			
		}
		finally {
			return resolve(pie_response);  
		}

	});
}

export const clickSearchElementModal = async(browser: puppeteer.Browser, url: string, searchElement: any, timeout: number, searchQuery: string) : Promise<ClickSearchElement> => {

	return new Promise(async (resolve, reject) => {

		const pie_response = {					
			querySelector: undefined,
			url: '',
			statusCode: -1,
			responseHeaders: undefined,				
			requestsChain: {},     
			listeners: [],
			requestInterceptions: [],
			responseInterceptions: [],
			hyperlinks: [],
			searchElement: undefined,         				
			search_url: null,
			status: undefined,
			exceptions: []
		}  as ClickSearchElement

		const timeoutId = setTimeout(() => {
            return reject();
        }, 300000);

        const executionContexts = ['execution_prepQuerySelector'];
        let page: puppeteer.Page;
       
		try {
				
			pie_response['url'] = url;
			pie_response['searchElement'] = searchElement;

			const querySelector = pie_response['querySelector'] =  utils.buildQuerySelector(searchElement,'input');																														

			page = await browser.newPage();
			
			// handling page errors.
			page.on('error', e => {
				throw e;
			});
					
			if(timeout) {					
				page.setDefaultNavigationTimeout(timeout);
			}
			
			const userAgent = new user_agent({ deviceCategory: 'desktop' });			
			await page.setUserAgent(userAgent.toString());			
			await page.setViewport({width: userAgent.data.screenWidth, height: userAgent.data.screenHeight});
			
			// scripts parsed by chrome
			const scriptsParsed: Array<any> = [];

			// note: add trailing slash since chrome adds it
			if (!url.endsWith('/'))
				url = url + '/';
										
			executionContexts.push('execution_createCDPSession');

			const client = await page.target().createCDPSession();
			await client.send('Debugger.enable');
			await client.send('Network.enable');     

			client.on('Debugger.scriptParsed', (params) => {

			try {
				scriptsParsed.push({'_scriptId':params.scriptId, '_scriptParsedData': params});                                    
			} catch {
                
             }

			});
        
            // urls hold redirect chain
		    let requestsChain : {[key: string]: any} = {};
		    const requestInterceptions: Array<{[key: string]: any}> = [];
            const responseInterceptions: Array<{[key: string]: any}> = [];	
        
			client.on('Network.requestWillBeSent', (e) => {
				
				const requestId = e.requestId;

				if(!requestsChain[requestId])
					requestsChain[requestId] = {};

				// check if url redirected
				if (typeof e.redirectResponse != 'undefined') {
					
					if(!requestsChain[requestId]['redirects'])
						requestsChain[requestId]['redirects'] = [];

						requestsChain[requestId]['redirects'].push({
						redirect: true,				
						status: e.redirectResponse.status,
						url: e.request.url,
						loader: e.loaderId,
						documentURL: e.documentURL,				
						type: e.type,
						frameId: e.frameId
						
					});

				} else {
				
					requestsChain[requestId] = {
							redirect: false,													
							url: e.request.url,							
							documentURL: e.documentURL,							
							type: e.type,
							frameId: e.frameId							
					};
				}
			});
		
			executionContexts.push('execution_setRequestInterception');

			const rootDomainRegex = utils.generateRootDomainRegex(url);
			const navigationRequestContext: Array<string> = [];	

			await page.setRequestInterception(true);	
							
			page.on('request', request => {
	
				try 
				{    													         								
					const url = request.url();        					
					
					if(!url.match(rootDomainRegex)) {     
						const resourceType = request.resourceType();
						const isNavigationRequest = request.isNavigationRequest();	 
						requestInterceptions.push({
							'navigationContext': _.last(navigationRequestContext),
							'context': _.last(executionContexts),
							'url': url,
							'resourceType': resourceType,
							'isNavigationRequest': isNavigationRequest,
							'headers': request.headers(),		
							'payload': request.postData()
												
						});
					}             
									
					request.continue();
	
				} catch (e) {}
	
			});
				
			page.on('response', async resp => {
				try 
				{    
					const url = resp.url();	
					const resourceType = resp.request().resourceType();	

					if(!url.match(rootDomainRegex)) {    

						let headers = undefined;	
						let payload = undefined;
																	
						try {							
							headers = await resp.headers();	
							if(!Consts.REFERRERS_CRAWLER.IGNORED_PAYLOADS_BY_RESOURCE_TYPES.includes(resourceType))
								payload = await resp.text();															
						} catch {}	

						responseInterceptions.push({
							'context': _.last(executionContexts),
							'url': url,
							'resourceType': resourceType, 
							'headers': headers,
							'payload': payload
						});
					}					
				}
				catch(e){}
			});
				
			let response = undefined;
			
			try {

				navigationRequestContext.push('navigationRequestContext_onload');
				executionContexts.push('execution_onload');
				    							
				try {
					response = await page.goto(url, {waitUntil: 'networkidle2'}); 										
				} catch(e) {					
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE_NETWORK_IDLE, message: e.message, stack: e.stack } as IBrowserError);
					requestsChain = {};
					scriptsParsed.splice(0, scriptsParsed.length);					
					requestInterceptions.splice(0, requestInterceptions.length);
					responseInterceptions.splice(0, responseInterceptions.length);					
				}
				finally {

					if(!response)
						response = await page.goto(url);
				}
							
				if(response)
				{
					pie_response['statusCode'] = response.status();
					pie_response['responseHeaders'] = response.headers();	
				}
						

			} catch (e) {		
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.VISITING_PAGE, message: e.message, stack: e.stack } as IBrowserError);							
				throw(e);
			}

			try {
					
				executionContexts.push('execution_collectingHyperlinks');
				const hyperlinks = await page.$$eval('a', e=>e.map((a: HTMLAnchorElement)=>{
					return {
					 id: a.getAttribute('id'),
					 name: a.getAttribute('name'), 					  
					 className: a.getAttribute('class'),
					 href: a.getAttribute('href'),
					 innerText: a.innerText.replace(/^\s+|\s+$/gm,''),
					 rel: a.getAttribute('rel'),
					 referrerpolicy: a.getAttribute('referrerpolicy')
				   };
			   }));

			   if(hyperlinks && hyperlinks.length > 0)
			   		pie_response['hyperlinks'] = hyperlinks;
			  
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACTING_LINKS, message: e.message, stack: e.stack } as IBrowserError);			
			}
																			
			const searchInputValue = searchQuery;     	
			executionContexts.push('execution_makingElementsVisible');	
			try {

				await page.$eval(querySelector, (element, selector) => {
					
					try { 
						                           						                     
						let els = document.querySelectorAll(selector);						                                                   
						[].forEach.call(els, (el: HTMLElement ) => {       							                                                                                       
							el.style.display = 'block';
							el.style.visibility = 'visible';
							el.style.zIndex = '999999999999';                                       
							while (el.parentNode) {                            
								el = el.parentNode as HTMLElement;                                                         
								if (el.tagName === 'body')
								{
									return;
								}
								else
								{        
									if(el.style){
										el.style.display  = 'block';
										el.style.visibility = 'visible';
									}        
										
								} 
							}							
						});           
						
						return;
						
					} catch (e) {
						
					}																																																											
					}, `button[type="submit"], input[type*="submit"], ${querySelector}`);                    
				
			} catch (e) {				
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.SET_ELEMENT_VISIBLE, message: e.message, stack: e.stack } as IBrowserError);	
			}

			try {

				executionContexts.push('execution_interactingWithInput');	
				await page.type(querySelector, searchInputValue, {delay: 500});  
				
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.TYPING_ON_ELEMENT, message: e.message, stack: e.stack } as IBrowserError);
			}
			
			try {
				
				executionContexts.push('execution_eventListenersAnalysis');	

				let context_listeners = await puppeteer_utils.performInputEventListenersAnalysis(client, querySelector, scriptsParsed);
				
				pie_response['listeners'].push({
					'context_listeners': context_listeners,
					'querySelector': querySelector,
					'inputElement': searchElement
				});       
							
			} catch (e) {				
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CAPTURING_EVENT_LISTENERS, message: e.message, stack: e.stack } as IBrowserError);				
			}
								
			try {     

				navigationRequestContext.push('navigationRequestContext_onsearch');
				executionContexts.push('execution_searchClick');	

				await Promise.all([
					page.type(querySelector, String.fromCharCode(13)),
					page.waitForNavigation({waitUntil: ['networkidle2', 'domcontentloaded']})
				]);
															
			} catch(e) {											
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.WAITING_FOR_SEARCH_RESULTS_NAVIGATION, message: e.message, stack: e.stack } as IBrowserError);
			}   
				
			try {

				executionContexts.push('execution_searchClickUrlExtraction');	

				pie_response['search_url'] = await Promise.race([page.evaluate(async ()=> { return window.location.href;}),
					page.waitFor(200000)
				]);
																
			} catch (e) {					
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACT_SEARCH_RESULTS_URL, message: e.message, stack: e.stack } as IBrowserError);
			}   
			
			finally {

				try {
					if(!pie_response['search_url'])					
						pie_response['search_url'] = page.url();										
				} catch (e) {
					pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.EXTRACT_SEARCH_RESULTS_URL, message: e.message, stack: e.stack } as IBrowserError);
                }   
			}
																
			pie_response['requestsChain'] = requestsChain;									
			pie_response['requestInterceptions'] = requestInterceptions;
			pie_response['responseInterceptions'] = responseInterceptions;		
			
			try {						

				executionContexts.push('execution_closingPage');
				await page.close();					
						
			} catch (e) {
				pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLOSING_PAGE, message: e.message, stack: e.stack } as IBrowserError);
			}
																	
			pie_response.status = { error: 0 };					

		} catch(e) {					
			pie_response['exceptions'].push({type: Consts.REFERRERS_CRAWLER.HEADLESS_BROWSER_EXCEPTION_TYPES.CLICK_STEP_EXCEPTION, message: e.message, stack: e.stack } as IBrowserError);
			pie_response.status = { error: 1, message: e.message, stack: e.stack, context: _.last(executionContexts) };			
		}
		finally {
			return resolve(pie_response);  
		}

	});
}

//document.querySelectorAll(`div[id*="modal"]`)