import fs from 'fs';
import _ from 'lodash';
import fetch from 'node-fetch';
import crypto from 'crypto';
import shell from 'shelljs';
import {parse, tldExists, getPublicSuffix} from 'tldjs';


export const extractTLD = (url: string) => {
  try {
    return tldExists(url) ? getPublicSuffix(url) : undefined;
  } catch {
    return undefined;
  }
}

export const extractHostname = (url: string) => {
  try {
    return parse(url)['hostname'];
  } catch {
    return undefined;
  } 
}

export const extractRootDomain = (url: string) => {
  try {
    return parse(url)['domain']; 
  } catch  {
    return undefined;
  }  
}

export const generateLogDir = (root_dir:string, url:string) => {
    
  const domainRootName = extractRootDomain(url);
  const ouputdir = root_dir + '/' +domainRootName+ '/';                            
  shell.mkdir('-p', ouputdir);

  return ouputdir;

}

export const generateRootDomainRegex = (url:string): RegExp => {
  return new RegExp(extractRootDomain(url));
}

export async function readFromFileAsync(fileName:string): Promise<string> {
    return new Promise((resolve, reject) => {
      fs.readFile(fileName, 'utf8', (err, data) => {
        if (err) reject(err);
        else resolve(data);
      });
    });
}

export function readFromFile(fileName: string): string { 
   return fs.readFileSync(fileName, 'utf8');
}

export function readJSONFromFile(fileName:string): string { 
  try {
    return JSON.parse(fs.readFileSync(fileName, 'utf8'));
  } catch { return null; } 
}

export async function writeToFileAsync(fileName: string, data: any): Promise<void> {
    return new Promise((resolve, reject) => {
      fs.writeFile(fileName, JSON.stringify(data, null, 4), err => {
        if (err) reject(err);
        else resolve();
      });
    });

}

export async function downloadFileAsync(uri: string, filename: string) : Promise<any> {
  
  try {
  
    const res = await fetch(uri);
      
    await new Promise((resolve, reject) => {

      try {
        const fileStream = fs.createWriteStream(filename);    
        res.body.pipe(fileStream);
        res.body.on("error", (err: any) => {
          reject(err);
        });
        fileStream.on("finish", function() {
          resolve();
        });  
      } catch (error) {
        reject(error);
      }
      
    });

    return res || undefined;

  } catch (error) {
    throw(error);
  }  
}

export async function getHeadersValue(uri: string, header: string) : Promise<any> {
  
  const res = await fetch(uri);

  let headersValue: any = undefined;

  if(res && res.headers)  
    headersValue = res.headers.get(header);

  return headersValue;

}

export async function preflightRequest(uri: string, timeout: number): Promise<any> {
  
  return new Promise(async(resolve, reject) => {
    
    const preflightDetails = {
      headers: <any> {},
      status: <number> undefined,
      timeout: 0,
      error: <any> undefined
    }   

    try {
      const timeoutId = setTimeout(() => {
        preflightDetails.timeout = 1;
        return resolve(preflightDetails);
      }, timeout);
  
      const res = await fetch(uri, {method: 'GET'});
    
      if(res && res.headers){
        res.headers.forEach((val , name) => {
          preflightDetails.headers[name] = val;
        });
        
      }   
        
      if(res && res.status)
        preflightDetails.status = res.status
        
      return resolve(preflightDetails);
    } catch (error) {     
      preflightDetails.error = error; 
      return resolve(preflightDetails);
    }
  })
};

export function generateHash(input: any) : string { 
  return crypto.createHash('md5').update(input.toString()).digest("hex");
}

export function sum(a: number, b: number): number {
  return a + b;
}

export function isUrl(url: string) : boolean {
  try {
      new URL (url);
      return true;   
  } catch {
      return false;
  }    
}

export function differenceWith(array_1: Array<any>, array_2: Array<any>): Array<any> {
  try {
    return _.differenceWith(array_1, array_2, _.isEqual); 
  } catch (err) {
      throw(err);
  }  
}

export function removeFromArrayList(item: any, array_1: Array<any>) : Array<any> {
  return _.pull(array_1, item); 
}

export async function removeFromArrayListAsync(item: any, array_1: Array<any>) : Promise<any> {
  return new Promise((resolve, reject) => {
    try {
      return resolve(_.pull(array_1, item));
    } catch (err) {
      return(reject(err));
    }
  });
}

export function trimStringSpaces(input: string) : string | undefined {
  return typeof input === 'string' ? input.trim().replace(/\s\s+/g, ' ') : undefined;
}

export const isDataUrl = (s: string) : boolean  => { 
  // https://developer.mozilla.org/en-US/docs/Web/HTTP/Basics_of_HTTP/Data_URIs 
  return !!s.match(/^\s*data:([a-z]+[\/\-][a-z0-9\-\+]+(;[a-z\-]+\=[a-z0-9\-]+)?)?(;base64)?,[a-z0-9\!\$\&\'\,\(\)\*\+\,\;\=\-\.\_\~\:\@\/\?\%\s]*\s*$/i);
  
}

export const redactDataURLWithHash = (url: string) : string | undefined => {
  
  try {
    
    const mediatypeRegexMatch = url.match(/^\s*data:([a-z]+[\/\-][a-z0-9\-\+]+(;[a-z\-]+\=[a-z0-9\-]+)?)?(;base64)?,/i);

	  if(mediatypeRegexMatch && mediatypeRegexMatch.length > 0)
	  {
		  const mediatype = mediatypeRegexMatch[0];
		  const hash = generateHash(url.replace(mediatype, ''));
		  return `${mediatype}hash:${hash}`;          		        
    }
    
  } catch(err) {
     throw(err);     
  }  
	
	return url;
}

export const getKBytes = (input: string, encoding: BufferEncoding = 'utf8') : number => {
  return Buffer.byteLength(input, encoding) / 1024;
}

export const extractFirstTextWithinQuotes = (input:string) : string => {
  const matches = input.match(/"(.*?)"/);
  return matches
    ? matches[1]
    : input;
}

export const extractLanguageFromHTML = (html: string) : string => { 

  try {
    
    const matched = html.match(/lang="([^\'\"]+)"/gmi); 
    if(matched && matched.length > 0)
      return extractFirstTextWithinQuotes(matched[0]);

  } catch {
    return undefined;  
  }
  
}

export const removeDuplicates = (data: Array<any>) : Array<any> => {
  
  if(!data || !data.length)
    return undefined;

  return data.reduce((a, b) => {
    if(a.indexOf(b) < 0) a.push(b);
    return a;
  }, []);

}

export const buildQuerySelector = (element: HTMLInputElement, type: string) => {

  let qs = '';

  try {

      if(element['type'] != 'text' && element['type'] != 'search')
          return '';

      if(element['id'] && typeof element['id'] != null && element['id'].length > 0)       
          qs = `[id="${element['id']}"]`; //improvement over qs = '#' + element['id']; which fails on numbers.
      else
          if(element['name'] && typeof element['name'] != null && element['name'].length > 0)
              qs = `${type}[name="${element['name']}"]`;
      else
          if(element['placeholder'] && typeof element['placeholder'] != null && element['placeholder'].length > 0)               
                qs = `${type}[placeholder="${element['placeholder']}"]`;
      else        
          if(element['className'] && typeof element['className'] != null && element['className'].length > 0)
          {
              qs = '';
              let classBuilder = element['className'].split(' ');            
              for(let cbitem of classBuilder)            
                  if(cbitem.length > 0)
                      qs=qs+'.'+cbitem;                        
          }
      else
          if(element['type'] && typeof element['type'] != null && element['type'].length > 0)          
            qs = `${type}[type="${element['type']}"]`;

  } catch (error) {       
      return null;
  }         
  return qs;
}

export const buildGenericQuerySelector = (element: HTMLInputElement, tagName: string) => {

  let qs = '';

  try {

      if(element['id'] && typeof element['id'] != null && element['id'].length > 0)       
          qs = `[id="${element['id']}"]`; //improvement over qs = '#' + element['id']; which fails on numbers.
      else
          if(element['name'] && typeof element['name'] != null && element['name'].length > 0)
              qs = `${tagName || ''}[name="${element['name']}"]`;
      else
          if(element['placeholder'] && typeof element['placeholder'] != null && element['placeholder'].length > 0)               
                qs = `${tagName || ''}[placeholder="${element['placeholder']}"]`;
      else        
          if(element['className'] && typeof element['className'] != null && element['className'].length > 0)
          {
              qs = `${tagName || ''}`;
              let classBuilder = element['className'].split(' ');            
              for(let cbitem of classBuilder)            
                  if(cbitem.length > 0)
                      qs=qs+'.'+cbitem;                        
          }
      else
          if(element['type'] && typeof element['type'] != null && element['type'].length > 0 && element['type'] != 'text')          
            qs = `${tagName}[type="${element['type']}"]`;

  } catch (error) {       
      return null;
  }         
  return qs;
}

export const validateSearchElementProps = (el: HTMLElement) => {

  const propValues = Object.values(el);
 
  for(const propVal of propValues)
  {
    if(propVal && propVal.length > 0)    
      return true;         
  }

  return false;
}