import puppeteer from 'puppeteer';


//lunches a new Chromium browser instance.
export const getBrowser = async (options?: puppeteer.LaunchOptions) => {

    try {
        
        if (!options) {
            options = getBrowserConfiguration();
        }
    
        let browser = await puppeteer.launch(options);
        return browser;
        
    } catch (error) {
       throw(error);
    }
}

// get browser config for puppeteer
export const getBrowserConfiguration = () => {
    return {
        headless: true,
        args: ['--enable-features=NetworkService', '--no-sandbox=true', '--window-size=1900,1050', '--disable-notifications',
        '--ignore-certificate-errors', '--disable-dev-shm-usage=true', '--disable-setuid-sandbox=true', '--disable-accelerated-2d-canvas=true', '--disable-gpu=true'],
        ignoreHTTPSErrors: true
    }
}

// extracts event listeners given a list of queryselectors and a list of script IDs (for matching purposes).
export const performInputEventListenersAnalysis = async (devtoolsProtocolClient: puppeteer.CDPSession, querySelector: string, scriptsParsed: Array<any>) => {
    
    try {
        
        let eventListeners = [];

        //e.g.: let exp = `document.querySelector("`+"#loginButton"+`")`;
        let exp = querySelector.indexOf('"') === -1 ? `document.querySelector("`+querySelector+`")` : `document.querySelector('`+querySelector+`')` ;
        
        //e.g.: const evalRes =  await devtoolsProtocolClient.send('Runtime.evaluate',{ expression:  'document'});
        const evalRes: any =  await devtoolsProtocolClient.send('Runtime.evaluate',{ expression:  exp});
        
        //get the listeners list.
        const listeners: any = await devtoolsProtocolClient.send('DOMDebugger.getEventListeners', { objectId: evalRes.result.objectId, pierce: true });
        
        if (listeners) {           
            // iterate through the listeners list.
            for (let i in listeners['listeners']) {
                let listener = listeners['listeners'][i];
                for (let j = 0; j < scriptsParsed.length; j++) {
                    if (scriptsParsed[j]['_scriptId'] == listener.scriptId) {
                        eventListeners.push({'scriptDetails':  scriptsParsed[j]['_scriptParsedData'], 'eventDetails': listener});                     
                    }
                }
            }
        }

    
    return eventListeners;

	}
	catch(e) { 
        throw(e)
    }
}

// the actual script source code.
export const getScriptSources = async (devtoolsProtocolClient: puppeteer.CDPSession, scriptsParced: Array<any>) => {    

    const results: Array<any> = [];

    try {
            
        scriptsParced.forEach(async (scriptParsed) => {
            try {
                const data = await devtoolsProtocolClient.send('Debugger.getScriptSource', { scriptId: scriptParsed._scriptId });
                results.push(data);   
            } catch {

            }                
        });

    } catch (e) {
         throw(e); 
    }

    return results;
}

// cleans up browser open pages.
export const closeOpenPages = async (browser: puppeteer.Browser): Promise<void> => {

    try {

        if(!browser)
            return;

        let pages = await browser.pages();	
        
        if(pages && pages.length)
        {
            pages.map(async (page) => {

                try {                    
                     await page.close();        
                } catch {
                    //todo
                } finally {
                    //todo
                }               
            });
        }
        	   
    } catch (error) {
        throw(error);
    }        
}